module net.BurtonRadons.parse.declaration;

private import net.BurtonRadons.parse.statement;

/** The base declaration object. */
class Declaration : Statement
{
    bit isExtern; /**< Defined to be extern; extern functions can't have bodies. */
    bit isAbstract; /**< Set on methods to indicate that it must be implemented by a subclass and it must have no body. */
    bit isFinal; /**< Set on methods to indicate that it cannot be overloaded by a subclass. */
    bit isStatic;
    /**< At the module level it indicates a symbol that is not visible to other modules.
       * At the aggregate level (a symbol in a class or struct) it indicates a symbol that is shared between all instances.
       * At the function body level it indicates a symbol with a permanent state.
       */
    bit isOverride; /**< Set on methods to indicate that this must be overriding a method in a parent class. */
    bit isConst; /**< Set on variables to indicate that their value cannot change.  const fields imply static. */

    alias int Convention;

    /** Calling convention types. */
    enum : Convention
    {
        D, /**< D linkage; callee cleans up stack, push parameters in left-to-right order, put rightmost parameter in eax. */
        C, /**< C linkage; caller cleans up stack, float promoted to double, push parameters in right-to-left order; mangle names with a prefixed underline. */
        Windows, /**< Windows linkage; callee cleans up stack, float promoted to double, push parameters in left-to-right order; mangle names by prefixing underline and appending "@" and the number of bytes in the arguments. */
        Pascal, /**< Pascal linkage; callee cleans up stack, float promoted to double, push parameters in left-to-right order. */
    }

    Convention convention; /**< The calling convention of this declaration. */

    Statement semantic0 (Scope scope) { return this; }
    Statement semantic1 (Scope scope) { return this; }
    Statement semantic2 (Scope scope) { return this; }

    /** Return whether in the calling convention the declaration uses, the 
      * caller cleans up the stack (true) or the one called cleans up the
      * stack (false).
      */

    bit callerCleansUpStack ()
    {
        if (convention == C)
            return true;
        if (convention == D || convention == Windows || convention == Pascal)
            return false;
        assert (0);
    }

    /** Whether the current calling convention pushes arguments from left-to-right order (true) or right-to-left order (false). */
    bit pushArgumentsLeftToRight ()
    {
        if (convention == D || convention == Windows || convention == Pascal)
            return true;
        if (convention == C)
            return false;
        assert (0);
    }

    /** Return whether the calling convention casts float arguments to doubles. */
    bit castFloatToDouble ()
    {
        if (convention == C)
            return true;
        if (convention == D || convention == Windows || convention == Pascal)
            return false;
        assert (0);
    }
}

/** Import declaration; imports this set of modules. */
class ImportDeclaration : Declaration
{
    Symbol [] names; /**< The modules to import. */
}

/** Aggregate type. */
class AggregateDeclaration : Declaration
{
    Symbol supertype; /**< Parent type. */
    Symbol name; /**< Type symbol or null. */

    Field [] fields; /**< The fields in the aggregate (semantic0). */
    int fieldSize = -1; /**< The total size of the fields. */
    Symbol [] methods; /**< The methods in the aggregate (semantic0). */
    Symbol [] vtable;
    bit semantic1done; /**< Set when the semantic1 method has already been called. */

    /** An aggregate field. */
    struct Field
    {
        int offset; /**< Byte offset from the start of the declaration (semantic2). */
        ubyte mask = 1; /**< Bit mask for bit fields (semantic2). */
        Symbol symbol; /**< Symbol pointer (semantic0). */
    }

    /** Get the offset of a field. */
    int fieldOffset (Symbol symbol)
    {
        for (int c; ; c ++)
        {
            assert (c < fields.length);
            if (fields [c].symbol === symbol)
                return fields [c].offset;
        }
    }

    /** Sum up this size and the size of the parents, or return -1 if one of them is incomplete. */
    int cellSize ()
    {
        return fieldSize;
    }

    override Statement semantic0 (Scope parent)
    {
        vtable ~= null; // pointer to ClassInfo.

        if (name !== null)
            name.decl = this;

        /* Install the fields and methods. */
        for (int c; c < scope.list.length; c ++)
            findContents (scope.list [c]);

        if (supertype !== null)
            supertype = supertype.semantic0 (parent);
        scope.parent = parent;
        parent.define (name);
        if (name !== null)
            name = name.semantic0 (scope);
        scope.semantic0 ();

        return this;
    }

    override Statement semantic1 (Scope parent)
    {
        scope.parent = parent;
        if (name !== null)
            name = name.semantic1 (scope);

        if (supertype !== null)
        {
            ClassDeclaration decl;
            Symbol value;

            value = supertype = parent.find (supertype.semantic1 (parent).name);
            assert (value !== null);
            decl = cast (ClassDeclaration) value.decl;
            assert (decl !== null);

            this.fields = decl.fields.dup ~ this.fields;
            this.methods = decl.methods.dup ~ this.methods;
        }

        fieldSize = 0;
        for (int c; c < fields.length; c ++)
        {
            int size = fields [c].symbol.type.cellSize ();

            if (size == -1)
            {
                fieldSize = -1;
                break;
            }

            if (size >= 4)
                fieldSize = (fieldSize + 3) / 4 * 4;
            else if (size == 2)
                fieldSize = (fieldSize + 1) / 2 * 2;
            fields [c].offset = fieldSize;
            fieldSize += size;
        }

        scope.semantic1 ();
        return this;
    }

    override Statement semantic2 (Scope parent)
    {
        scope.parent = parent;
        if (name !== null)
            name = name.semantic2 (scope);
        if (supertype !== null)
            supertype = supertype.semantic2 (parent);

        fieldSize = 0;
        for (int c; c < fields.length; c ++)
        {
            int size = fields [c].symbol.type.cellSize ();

            assert (size != -1);

            if (size >= 4)
                fieldSize = (fieldSize + 3) / 4 * 4;
            else if (size == 2)
                fieldSize = (fieldSize + 1) / 2 * 2;
            fields [c].offset = fieldSize;
            fieldSize += size;
        }

        scope.semantic2 ();
        return this;
    }

    /* Setup fields. */
    void findContents (Statement base)
    {
        if (cast (VariableDeclaration) base)
        {
            VariableDeclaration stat = cast (VariableDeclaration) base;
            Field field;

            stat.aggr = this;
            if (stat.isExtern)
                return;
            for (int c; c < stat.list.length; c ++)
            {
                field.symbol = stat.list [c];
                fields ~= field;
            }
        }
        else if (cast (FunctionDeclaration) base)
        {
            FunctionDeclaration stat = cast (FunctionDeclaration) base;
            
            vtable ~= stat.name;

            stat.aggr = this;
            if (!stat.isStatic)
                stat.hasThis = true;
            methods ~= stat.name;
        }
        else
            throw new Error ("AggregateDeclaration.findContents (" ~ base.classinfo.name ~ ")");
    }

    int methodIndex (Symbol symbol)
    {
        for (int c; ; c ++)
        {
            assert (c < vtable.length);
            if (vtable [c] === symbol)
                return c;
        }
    }
}

/** Struct declaration. */
class StructDeclaration : AggregateDeclaration
{
    import net.BurtonRadons.parse.type;
    
    override Statement semantic0 (Scope parent)
    {
        if (name === null)
            name = new Symbol ();
        name.type = new StructType (this);
        return super.semantic0 (parent);
    }
}

/** Class declaration. */
class ClassDeclaration : AggregateDeclaration
{
    import net.BurtonRadons.parse.type;
    
    override int fieldOffset (Symbol symbol)
    {
        return super.fieldOffset (symbol) + 8;
    }

    override int cellSize ()
    {
        return 4;
    }

    int contentSize ()
    {
        return super.cellSize ();
    }

    override Statement semantic0 (Scope parent)
    {
        assert (name !== null);
        name.type = new ClassType (this);
        return super.semantic0 (parent);
    }

    override Statement semantic1 (Scope parent)
    {
        Statement stat = super.semantic1 (parent);

        if (cast (ClassDeclaration) stat)
            (cast (ClassDeclaration) stat).fieldSize += 8;
        return stat;
    }

    override Statement semantic2 (Scope parent)
    {
        Statement stat = super.semantic2 (parent);

        if (cast (ClassDeclaration) stat)
            (cast (ClassDeclaration) stat).fieldSize += 8;
        return stat;
    }
}

/** Enum declaration. */
class EnumDeclaration : Declaration
{
    import net.BurtonRadons.parse.type;
    
    Type type; /**< The type of the entries, null for default. */
    Symbol name; /**< Defined name or null for an unnamed enum. */
    Symbol [] list; /**< Values in the enumeration. */

    override Statement semantic0 (Scope scope)
    {
        if (type === null)
            type = IntType.singleton;

        if (name !== null)
        {
            name.type = type;
            name.decl = this;
            scope.decls [name.name] = name;
        }
        else
        {
            for (int c; c < list.length; c ++)
            {
                list [c].type = type;
                scope.decls [list [c].name] = list [c];
            }
        }

        return this;
    }
}

/** A value alias declaration. */
class AliasDeclaration : Declaration
{
    import net.BurtonRadons.parse.type;
    
    Type type; /**< The type to alias. */
    Symbol name; /**< The type name. */

    override Statement semantic0 (Scope scope)
    {
        name.decl = this;
        name.type = type;
        scope.decls [name.name] = name;
        return this;
    }
}

/** A function declaration, possibly with body. */
class FunctionDeclaration : Declaration
{
    import net.BurtonRadons.parse.error;
    import net.BurtonRadons.parse.parser;
    import net.BurtonRadons.parse.type;
    
    Type type; /**< Type descriptor. */
    Symbol name; /**< Name of the function. */

    AggregateDeclaration aggr; /**< Aggregate this is a member function of or null. */
    bit hasThis; /**< Set if this function has a "this" pointer. */
    Statement inStatement; /* The "in" contract or null. */
    Statement outStatement; /* The "out" contract or null. */
    Symbol outReturn; /* The name of the return value for the "out" contract or null. */
    Statement bodyStatement; /* The function body or null. */

    this (Parser parser, Marker mark)
    {
        this.mark = mark;
        scope = new Scope (parser); /* Scope holds the parameter names. */
        scope.root = this;
    }

    /** Return whether this is a function nested in another function. */
    bit isNested ()
    {
        return nestingFunction () !== null;
    }

    /** Return the nested function scope or null. */
    FunctionDeclaration nestingFunction ()
    {
        return cast (FunctionDeclaration) scope.parent.findRoot ();
    }

    override Statement semantic0 (Scope parent)
    {
        scope.parent = parent;
        parent.decls [name.name] = name;

        FunctionType type = cast (FunctionType) this.type;

        type.decl = this;

        for (int c; c < type.args.length; c ++)
        {
            if (type.args [c].name !== null)
                scope.decls [type.args [c].name] = type.args [c];
        }

        name.decl = this;
        name.type = this.type;
        if (inStatement !== null)
        {
            inStatement.parent = this;
            inStatement = inStatement.semantic0 (scope);
        }
        if (outStatement !== null)
        {
            outStatement.parent = this;
            outStatement = outStatement.semantic0 (scope);
        }
        if (bodyStatement !== null)
        {
            bodyStatement.parent = this;
            bodyStatement = bodyStatement.semantic0 (scope);
        }

        if (isAbstract && bodyStatement !== null)
        {
            scope.semanticError (new AbstractWithBodyError (mark, name.name));
            isAbstract = false;
        }

        return this;
    }

    override Statement semantic1 (Scope parent)
    {
        if (isNested ())
            hasThis = true;
        if (inStatement !== null)
            inStatement = inStatement.semantic1 (scope);
        if (outStatement !== null)
            outStatement = outStatement.semantic1 (scope);
        if (bodyStatement !== null)
            bodyStatement = bodyStatement.semantic1 (scope);
        return this;
    }

    override Statement semantic2 (Scope parent)
    {
        if (inStatement !== null)
            inStatement = inStatement.semantic2 (scope);
        if (outStatement !== null)
            outStatement = outStatement.semantic2 (scope);
        if (bodyStatement !== null)
            bodyStatement = bodyStatement.semantic2 (scope);
        return this;
    }
}

/** A constructor function declaration. */
class ConstructorDeclaration : FunctionDeclaration
{
    this (Parser parser, Marker mark)
    {
        super (parser, mark);
    }
}

/** A unittest function declaration. */
class UnittestDeclaration : FunctionDeclaration
{
    this (Parser parser, Marker mark)
    {
        super (parser, mark);
    }
}

/** A variable declaration. */
class VariableDeclaration : Declaration
{
    import net.BurtonRadons.parse.parser;
    import net.BurtonRadons.parse.type;
    
    Symbol [] list; /**< Variables this defines. */
    AggregateDeclaration aggr; /**< Aggregate this belongs to or null. */
    Parser parser; /**< Parser this is from. */

    /** Assign the parameter. */
    this (Parser parser)
    {
        this.parser = parser;
    }

    override Statement semantic0 (Scope scope)
    {
        for (int c; c < list.length; c ++)
        {
            Symbol item = list [c];

            scope.decls [item.name] = item;

            if (cast (FunctionType) item.type)
                (cast (FunctionType) item.type).decl = this;
            if (item.init !== null)
                item.init = item.init.semantic0 (scope);
            item.decl = this;
        }
        return this;
    }

    override Statement semantic1 (Scope scope)
    {
        for (int c; c < list.length; c ++)
        {
            Symbol item = list [c];

            item = item.semantic1 (scope);

            /* Give default initialisers. */
            if (item.init === null && !isExtern)
                item.init = item.type.field (mark, "init");

            list [c] = item;
        }

        return this;
    }

    override Statement semantic2 (Scope scope)
    {
        for (int c; c < list.length; c ++)
        {
            list [c] = list [c].semantic2 (scope);
            if (list [c].init !== null)
                list [c].init = list [c].init.implicitCastTo (scope, list [c].type);
        }
        return this;
    }
}
